<?php
// views/user_detail.php
// View ini digunakan untuk menampilkan detail profil pengguna.
// Variabel yang diharapkan:
// - $user (array asosiatif data pengguna)
// - $pemesanan_list (array asosiatif daftar pemesanan pengguna)
// Variabel-variabel ini disediakan oleh UserController->detail()

// Pesan sukses atau error dari sesi
if (isset($_SESSION['error_message'])) {
    echo "<p class='message error'>" . htmlspecialchars($_SESSION['error_message']) . "</p>";
    unset($_SESSION['error_message']);
}
if (isset($_SESSION['success_message'])) {
    echo "<p class='message success'>" . htmlspecialchars($_SESSION['success_message']) . "</p>";
    unset($_SESSION['success_message']);
}

// Pastikan $user tidak kosong sebelum mencoba mengakses datanya
if (!isset($user) || !is_array($user)) {
    echo "<p class='message error'>Data pengguna tidak ditemukan.</p>";
    return; // Hentikan eksekusi jika data user tidak ada
}
?>

<div class="detail-container">
    <h1 class="page-title">Detail Profil Pengguna</h1>
    <p class="page-description">Informasi lengkap dan riwayat pemesanan untuk pengguna ini.</p>

    <div class="profile-header">
        <?php
        $profile_pic_path = isset($user['foto']) && $user['foto'] ? "assets/profile/" . htmlspecialchars(basename($user['foto'])) : null;
        ?>
        <?php if ($profile_pic_path && file_exists(dirname(__DIR__) . '/' . $profile_pic_path)): ?>
            <img src="<?= $profile_pic_path ?>" alt="Foto Profil" class="profile-avatar">
        <?php else: ?>
            <div class="profile-avatar no-avatar">
                <svg xmlns="http://www.w3.org/2000/svg" width="60" height="60" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-user"><path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path><circle cx="12" cy="7" r="4"></circle></svg>
            </div>
        <?php endif; ?>
        <div class="profile-info">
            <h2><?php echo htmlspecialchars($user['fullname']); ?></h2>
            <p class="username-role"><?php echo htmlspecialchars($user['username']); ?> (<?php echo htmlspecialchars(ucfirst($user['role'])); ?>)</p>
        </div>
        <div class="profile-actions">
            <a href="index.php?hal=user_edit&id=<?php echo htmlspecialchars($user['id']); ?>"
               class="action-btn action-btn-edit">
                Edit Profil
            </a>
            <?php if (isset($_SESSION['user_role']) && $_SESSION['user_role'] === 'admin' && ($_SESSION['user_id'] ?? null) != $user['id']): ?>
            <form action="index.php?hal=user_destroy&id=<?php echo htmlspecialchars($user['id']); ?>" method="POST"
                    class="delete-form" onsubmit="return confirm('Apakah Anda yakin ingin menghapus pengguna ini? Tindakan ini tidak dapat dibatalkan jika sudah ada pemesanan.');">
                <button type="submit" class="action-btn action-btn-danger">
                    Hapus Pengguna
                </button>
            </form>
            <?php endif; ?>
        </div>
    </div>

    <hr class="divider">

    <div class="info-section">
        <h3>Informasi Kontak</h3>
        <div class="contact-details">
            <p><i class="bi bi-envelope-fill icon-left"></i> <strong>Email:</strong> <?php echo htmlspecialchars($user['email']); ?></p>
            <p><i class="bi bi-phone-fill icon-left"></i> <strong>Nomor HP:</strong> <?php echo htmlspecialchars($user['no_hp'] ?? '-'); ?></p>
            <p><i class="bi bi-gender-<?php echo ($user['jenis_kelamin'] == 'L' ? 'male' : 'female'); ?> icon-left"></i> <strong>Jenis Kelamin:</strong> <?php echo htmlspecialchars($user['jenis_kelamin'] == 'L' ? 'Laki-laki' : 'Perempuan'); ?></p>
        </div>
    </div>

    <hr class="divider">

    <h3 class="section-title-center">Riwayat Pemesanan</h3>
    <?php if (!empty($pemesanan_list)): ?>
        <div class="table-container">
            <table class="styled-table">
                <thead>
                    <tr>
                        <th>No.</th>
                        <th>Film</th>
                        <th>Jadwal</th>
                        <th>Kursi</th>
                        <th class="text-right">Harga</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $no_pemesanan = 1; ?>
                    <?php foreach ($pemesanan_list as $pemesanan): ?>
                        <tr>
                            <td data-label="No."><?php echo $no_pemesanan++; ?></td>
                            <td data-label="Film"><?php echo htmlspecialchars($pemesanan['film_judul']); ?></td>
                            <td data-label="Jadwal">
                                <?php echo htmlspecialchars($pemesanan['tanggal_tayang']) . ' ' . htmlspecialchars(substr($pemesanan['waktu_tayang'], 0, 5)); ?>
                            </td>
                            <td data-label="Kursi"><?php echo htmlspecialchars($pemesanan['kursi_nomor']); ?></td>
                            <td data-label="Harga" class="text-right">Rp <?php echo number_format($pemesanan['harga_saat_pemesanan'], 0, ',', '.'); ?></td>
                            <td data-label="Status"><?php echo htmlspecialchars(ucfirst($pemesanan['status_pemesanan'])); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php else: ?>
        <div class="empty-state">
            <svg xmlns="http://www.w3.org/2000/svg" width="60" height="60" viewBox="0 0 24 24" fill="none" stroke="#ccc" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-credit-card"><rect x="1" y="4" width="22" height="16" rx="2" ry="2"></rect><line x1="1" y1="10" x2="23" y2="10"></line></svg>
            <p>Pengguna ini belum memiliki riwayat pemesanan tiket.</p>
        </div>
    <?php endif; ?>
</div>

---

<style>
    /* Global/Base Styles (dari halaman film/user list) */
    body {
        font-family: 'Poppins', sans-serif;
        background-color: #f0f2f5;
        color: #333;
        line-height: 1.6;
        margin: 0;
        padding: 0;
    }

    /* Message Styling (Success/Error) - dari halaman film/user list */
    .message {
        padding: 12px 20px;
        margin: 20px auto;
        border-radius: 8px;
        text-align: center;
        font-weight: 600;
        max-width: 900px;
        box-sizing: border-box;
        animation: fadeInDown 0.5s ease-out;
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    @keyframes fadeInDown {
        from { opacity: 0; transform: translateY(-20px); }
        to { opacity: 1; transform: translateY(0); }
    }
    .message.error {
        background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;
    }
    .message.success {
        background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb;
    }

    /* Main Detail Container */
    .detail-container {
        max-width: 900px; /* Lebih lebar dari form, lebih sempit dari list */
        margin: 30px auto;
        padding: 30px;
        background-color: #ffffff;
        border-radius: 12px;
        box-shadow: 0 6px 20px rgba(0, 0, 0, 0.1);
        animation: fadeIn 0.8s ease-out;
    }
    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(20px); }
        to { opacity: 1; transform: translateY(0); }
    }

    .page-title {
        text-align: center;
        color: #1a2a3a;
        font-size: 2.8em;
        font-weight: 800;
        margin-top: 0;
        margin-bottom: 10px;
        position: relative;
        padding-bottom: 15px;
    }
    .page-title::after {
        content: '';
        width: 80px;
        height: 4px;
        background-color: #007bff;
        position: absolute;
        bottom: 0;
        left: 50%;
        transform: translateX(-50%);
        border-radius: 2px;
    }

    .page-description {
        text-align: center;
        color: #6a737d;
        font-size: 1.1em;
        margin-bottom: 30px;
        line-height: 1.5;
    }

    /* Profile Header Section */
    .profile-header {
        display: flex;
        flex-direction: column;
        align-items: center;
        text-align: center;
        margin-bottom: 25px;
        padding-bottom: 20px;
        border-bottom: 1px solid #eee;
    }

    .profile-avatar {
        width: 150px;
        height: 150px;
        border-radius: 50%;
        object-fit: cover;
        border: 4px solid #007bff; /* Warna border biru */
        margin-bottom: 15px;
        box-shadow: 0 5px 15px rgba(0, 123, 255, 0.2);
        transition: transform 0.3s ease;
    }
    .profile-avatar:hover {
        transform: scale(1.05);
    }

    .profile-avatar.no-avatar {
        background-color: #e0e0e0;
        display: flex;
        justify-content: center;
        align-items: center;
        font-size: 4em;
        color: #aaa;
        border: 4px dashed #ccc; /* Border putus-putus untuk no-avatar */
        box-shadow: none;
    }
    .profile-avatar.no-avatar svg {
        color: #aaa;
        width: 60px;
        height: 60px;
    }

    .profile-info h2 {
        margin: 0 0 5px;
        color: #1a2a3a;
        font-size: 2em;
        font-weight: 700;
    }

    .profile-info .username-role {
        color: #6a737d;
        font-size: 1em;
        margin: 0;
    }

    .profile-actions {
        margin-top: 20px;
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
        justify-content: center;
    }

    /* Divider */
    .divider {
        border: 0;
        border-top: 1px solid #e0e0e0;
        margin: 30px 0;
    }

    /* Info Section */
    .info-section {
        padding: 15px 0;
        /* text-align: center; // Dihapus untuk fleksibilitas ikon */
    }
    .info-section h3 {
        color: #1a2a3a;
        font-size: 1.8em;
        font-weight: 700;
        margin-bottom: 15px;
        position: relative;
        display: inline-block;
        padding-bottom: 5px;
        left: 50%;
        transform: translateX(-50%);
        white-space: nowrap; /* Mencegah teks judul terputus */
    }
    .info-section h3::after {
        content: '';
        width: 50%;
        height: 3px;
        background-color: #007bff;
        position: absolute;
        bottom: 0;
        left: 25%;
        border-radius: 2px;
    }

    .contact-details {
        display: flex;
        flex-direction: column;
        gap: 10px; /* Jarak antar item kontak */
        max-width: 400px; /* Batasi lebar untuk kerapihan */
        margin: 0 auto; /* Pusatkan blok contact details */
        padding: 15px;
        background-color: #f8fafd; /* Latar belakang sedikit berbeda */
        border-radius: 10px;
        border: 1px solid #eef2f6;
    }

    .contact-details p {
        font-size: 1.05em;
        margin: 0; /* Hapus margin default paragraf */
        color: #555;
        display: flex; /* Aktifkan flexbox untuk ikon dan teks */
        align-items: center; /* Pusatkan vertikal ikon dan teks */
    }
    .contact-details p strong {
        color: #333;
        margin-right: 5px; /* Sedikit jarak antara label dan nilai */
    }
    .contact-details .icon-left {
        margin-right: 10px; /* Jarak antara ikon dan teks */
        color: #007bff; /* Warna ikon */
        font-size: 1.2em; /* Ukuran ikon */
        min-width: 20px; /* Memastikan ikon punya ruang tetap */
        text-align: center;
    }

    /* Section Title Center for Riwayat Pemesanan */
    .section-title-center {
        text-align: center;
        color: #1a2a3a;
        font-size: 1.8em;
        font-weight: 700;
        margin-top: 20px;
        margin-bottom: 20px;
        position: relative;
        display: inline-block;
        padding-bottom: 5px;
        left: 50%;
        transform: translateX(-50%);
        white-space: nowrap; /* Mencegah teks judul terputus */
    }
    .section-title-center::after {
        content: '';
        width: 50%;
        height: 3px;
        background-color: #007bff;
        position: absolute;
        bottom: 0;
        left: 25%;
        border-radius: 2px;
    }

    /* Table Styling (dari halaman user list) */
    .table-container {
        overflow-x: auto;
        margin-top: 20px;
        background-color: #fff;
        border-radius: 12px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        padding: 20px;
    }

    .styled-table {
        width: 100%;
        border-collapse: collapse;
        margin: 0;
        font-size: 0.95em;
    }

    .styled-table th,
    .styled-table td {
        padding: 12px 15px;
        border: 1px solid #e0e0e0;
        text-align: left;
    }

    .styled-table thead tr {
        background-color: #007bff;
        color: white;
        text-align: left;
        font-weight: 600;
    }

    .styled-table tbody tr:nth-of-type(even) {
        background-color: #f6f6f6;
    }

    .styled-table tbody tr:hover {
        background-color: #e9f5ff;
        cursor: default;
    }

    .styled-table td.text-right {
        text-align: right;
    }

    /* Tombol Aksi - Warna Minimalis (dari halaman user list) */
    .action-btn {
        padding: 10px 15px;
        border-radius: 20px;
        text-decoration: none;
        font-weight: 600;
        font-size: 0.9em;
        transition: all 0.2s ease;
        text-align: center;
        white-space: nowrap;
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        border: 1px solid transparent;
        display: inline-block;
    }
    .action-btn-edit {
        background-color: #17a2b8;
        color: white;
        border-color: #17a2b8;
    }
    .action-btn-edit:hover {
        background-color: #138496;
        border-color: #138496;
        color: #e0e0e0;
        transform: translateY(-1px);
    }
    .action-btn-danger {
        background-color: #dc3545;
        color: white;
        border: none;
        cursor: pointer;
    }
    .action-btn-danger:hover {
        background-color: #c82333;
        transform: translateY(-1px);
    }

    .delete-form {
        display: inline-block;
        margin: 0;
        padding: 0;
    }

    /* Empty State (dari halaman user list) */
    .empty-state {
        text-align: center;
        padding: 50px 20px;
        background-color: #fefefe;
        border: 1px dashed #e0e0e0;
        border-radius: 10px;
        margin-top: 30px;
        color: #888;
        animation: popIn 0.5s ease-out;
    }
    @keyframes popIn {
        from { opacity: 0; transform: scale(0.9); }
        to { opacity: 1; transform: scale(1); }
    }
    .empty-state svg {
        margin-bottom: 20px;
        color: #ccc;
    }
    .empty-state p {
        font-size: 1.1em;
        margin-bottom: 25px;
    }

    /* Responsive Adjustments */
    @media (max-width: 768px) {
        .detail-container {
            margin: 20px 15px;
            padding: 20px;
        }
        .page-title {
            font-size: 2.2em;
        }
        .page-description {
            font-size: 1em;
        }
        .profile-header {
            padding-bottom: 15px;
        }
        .profile-info h2 {
            font-size: 1.8em;
        }
        .profile-info .username-role {
            font-size: 0.9em;
        }
        .profile-actions {
            flex-direction: column;
            align-items: stretch;
            gap: 8px;
        }
        .action-btn {
            width: 100%;
            min-width: unset;
        }
        .info-section h3, .section-title-center {
            font-size: 1.5em;
            transform: translateX(-50%); /* Re-center after font size change */
        }
        .info-section h3::after, .section-title-center::after {
             left: 25%; /* Ensure line remains centered */
        }
        .info-section p {
            font-size: 1em;
        }
        .contact-details {
            max-width: 100%; /* Lebar penuh di mobile */
            padding: 10px;
        }
        .contact-details p {
            font-size: 0.95em;
        }
        /* Mobile table styling for Riwayat Pemesanan (from user list) */
        .styled-table, .table-container {
            border-radius: 8px;
        }
        .styled-table thead {
            display: none;
        }
        .styled-table, .styled-table tbody, .styled-table tr, .styled-table td {
            display: block;
            width: 100%;
        }
        .styled-table tr {
            margin-bottom: 15px;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            padding: 10px;
            background-color: #fff;
        }
        .styled-table td {
            text-align: right !important;
            padding-left: 50% !important;
            position: relative;
            border: none;
        }
        .styled-table td::before {
            content: attr(data-label);
            position: absolute;
            left: 10px;
            width: calc(50% - 20px);
            padding-right: 10px;
            white-space: nowrap;
            text-align: left;
            font-weight: 600;
            color: #555;
        }
        /* Specific data-labels for pemesanan table */
        .styled-table td:first-child::before { content: "No."; }
        .styled-table td:nth-child(2)::before { content: "Film:"; }
        .styled-table td:nth-child(3)::before { content: "Jadwal:"; }
        .styled-table td:nth-child(4)::before { content: "Kursi:"; }
        .styled-table td:nth-child(5)::before { content: "Harga:"; }
        .styled-table td:last-child::before { content: "Status:"; }
    }
    @media (max-width: 480px) {
        .page-title {
            font-size: 2em;
            padding-bottom: 10px;
        }
        .page-title::after {
            width: 50px;
            height: 3px;
        }
        .profile-avatar {
            width: 120px;
            height: 120px;
        }
        .profile-avatar.no-avatar svg {
            width: 50px;
            height: 50px;
        }
        .profile-info h2 {
            font-size: 1.5em;
        }
        .profile-actions {
            margin-top: 15px;
        }
        .action-btn {
            font-size: 0.8em;
            padding: 8px 12px;
        }
        .info-section h3, .section-title-center {
            font-size: 1.3em;
        }
        .info-section h3::after, .section-title-center::after {
            width: 40%;
            left: 30%;
        }
        .info-section p {
            font-size: 0.95em;
        }
    }
</style>