<?php
// models/Film.php

// Pastikan BaseModel.php di-include. Sesuaikan path jika struktur direktori Anda berbeda.
require_once dirname(__DIR__) . '/models/BaseModel.php';

class Film extends BaseModel {
    // Properti yang mendefinisikan nama tabel di database
    protected $table = 'film';

    /**
     * Konstruktor untuk Film.
     * Menerima objek koneksi MySQLi dan meneruskannya ke BaseModel.
     * @param mysqli $conn Objek koneksi database MySQLi.
     */
    public function __construct(mysqli $conn) {
        parent::__construct($conn);
    }

    /**
     * Mengambil semua data film dari database.
     * Sekarang mengambil SEMUA KOLOM film (f.*) dan JOIN ke kategori/studio.
     * Ini untuk memenuhi kebutuhan views/home.php dan views/film_list.php
     * yang mungkin menampilkan detail seperti cover, sinopsis, kategori, dll.
     *
     * @return array Array asosiatif berisi semua data film.
     */
    public function getAllFilms() {
        $sql = "SELECT f.*, k.kategori, s.nama AS studio_nama
                FROM {$this->table} f
                JOIN kategori k ON f.kategori_id = k.id
                JOIN studio s ON f.studio_id = s.id
                ORDER BY f.judul ASC"; 
        
        $result = $this->conn->query($sql);
        if (!$result) {
            error_log("SQL Error in getAllFilms: " . $this->conn->error);
            return [];
        }
        $films = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $films[] = $row;
            }
        }
        return $films;
    }

    /**
     * Mengambil data film berdasarkan ID.
     * Termasuk informasi kategori dan studio.
     *
     * @param int $id ID film.
     * @return array|null Array asosiatif data film jika ditemukan, null jika tidak.
     */
    public function getFilmById($id) {
        $stmt = $this->conn->prepare("SELECT f.*, k.kategori, s.nama AS studio_nama
                                      FROM {$this->table} f
                                      JOIN kategori k ON f.kategori_id = k.id
                                      JOIN studio s ON f.studio_id = s.id
                                      WHERE f.id = ?");
        if (!$stmt) {
            error_log("Prepare failed in getFilmById: " . $this->conn->error);
            return null;
        }
        $stmt->bind_param("i", $id);
        if (!$stmt->execute()) {
            error_log("Execute failed in getFilmById: " . $stmt->error);
            return null;
        }
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }

    /**
     * Menambahkan film baru ke database.
     * Menggunakan nama 'createFilm' untuk konsistensi.
     *
     * @param string $judul Judul film.
     * @param string $tanggal_rilis Tanggal rilis (format YYYY-MM-DD).
     * @param string $sinopsis Sinopsis film.
     * @param string|null $cover Nama file cover film.
     * @param int $kategori_id ID kategori film.
     * @param int $studio_id ID studio yang menayangkan film ini.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function createFilm($judul, $tanggal_rilis, $sinopsis, $cover, $kategori_id, $studio_id) {
        // Asumsi kolom created_at dan updated_at ada di tabel 'film' dan diatur otomatis oleh DB.
        // Jika tidak, Anda perlu menambahkannya ke query INSERT ini dan bind_param.
        $stmt = $this->conn->prepare("INSERT INTO {$this->table} (judul, tanggal_rilis, sinopsis, cover, kategori_id, studio_id) VALUES (?, ?, ?, ?, ?, ?)");
        if (!$stmt) {
            error_log("Prepare failed in createFilm: " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("ssssii", $judul, $tanggal_rilis, $sinopsis, $cover, $kategori_id, $studio_id);
        
        if ($stmt->execute()) {
            return true;
        } else {
            error_log("Execute failed in createFilm: " . $stmt->error);
            return false;
        }
    }

    /**
     * Memperbarui data film yang sudah ada di database.
     * Menggunakan nama 'updateFilm' untuk konsistensi.
     *
     * @param int $id ID film yang akan diperbarui.
     * @param string $judul Judul film.
     * @param string $tanggal_rilis Tanggal rilis (format YYYY-MM-DD).
     * @param string $sinopsis Sinopsis film.
     * @param string|null $cover Nama file cover film.
     * @param int $kategori_id ID kategori film.
     * @param int $studio_id ID studio yang menayangkan film ini.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function updateFilm($id, $judul, $tanggal_rilis, $sinopsis, $cover, $kategori_id, $studio_id) {
        // Asumsi kolom updated_at diatur otomatis oleh DB (ON UPDATE CURRENT_TIMESTAMP).
        // Jika tidak, Anda perlu menambahkannya ke query UPDATE ini dan bind_param.
        $sql = "UPDATE {$this->table} SET judul = ?, tanggal_rilis = ?, sinopsis = ?, cover = ?, kategori_id = ?, studio_id = ? WHERE id = ?";
        $stmt = $this->conn->prepare($sql);
        if (!$stmt) {
            error_log("Prepare failed in updateFilm: " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("ssssiii", $judul, $tanggal_rilis, $sinopsis, $cover, $kategori_id, $studio_id, $id);
        
        if ($stmt->execute()) {
            return true;
        } else {
            error_log("Execute failed in updateFilm: " . $stmt->error);
            return false;
        }
    }

    /**
     * Menghapus film dari database berdasarkan ID.
     * Perhatikan FOREIGN KEY CONSTRAINTS. Jika ada jadwal atau pemesanan terkait film ini,
     * penghapusan mungkin akan diblokir kecuali Anda mengatur ON DELETE CASCADE di database.
     * Menggunakan nama 'deleteFilm' untuk konsistensi.
     *
     * @param int $id ID film yang akan dihapus.
     * @return bool True jika berhasil, false jika gagal.
     */
    public function deleteFilm($id) {
        $stmt = $this->conn->prepare("DELETE FROM {$this->table} WHERE id = ?");
        if (!$stmt) {
            error_log("Prepare failed in deleteFilm: " . $this->conn->error);
            return false;
        }
        $stmt->bind_param("i", $id);
        
        if ($stmt->execute()) {
            return true;
        } else {
            error_log("Execute failed in deleteFilm: " . $stmt->error);
            return false;
        }
    }

    /**
     * Mengambil semua film yang ditayangkan di studio tertentu.
     * Diperlukan untuk validasi saat menghapus studio.
     *
     * @param int $studio_id ID studio.
     * @return array Array asosiatif berisi data film di studio tersebut.
     */
    public function getFilmsByStudioId($studio_id) {
        $stmt = $this->conn->prepare("SELECT id FROM {$this->table} WHERE studio_id = ?");
        if (!$stmt) {
            error_log("Prepare failed in getFilmsByStudioId: " . $this->conn->error);
            return [];
        }
        $stmt->bind_param("i", $studio_id);
        if (!$stmt->execute()) {
            error_log("Execute failed in getFilmsByStudioId: " . $stmt->error);
            return [];
        }
        $result = $stmt->get_result();
        $films = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $films[] = $row;
            }
        }
        return $films;
    }

    /**
     * Mengambil semua film yang termasuk dalam kategori tertentu.
     * Diperlukan untuk validasi saat menghapus kategori.
     *
     * @param int $kategori_id ID kategori.
     * @return array Array asosiatif berisi data film dalam kategori tersebut.
     */
    public function getFilmsByKategoriId($kategori_id) {
        $stmt = $this->conn->prepare("SELECT id FROM {$this->table} WHERE kategori_id = ?");
        if (!$stmt) {
            error_log("Prepare failed in getFilmsByKategoriId: " . $this->conn->error);
            return [];
        }
        $stmt->bind_param("i", $kategori_id);
        if (!$stmt->execute()) {
            error_log("Execute failed in getFilmsByKategoriId: " . $stmt->error);
            return [];
        }
        $result = $stmt->get_result();
        $films = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $films[] = $row;
            }
        }
        return $films;
    }

    /**
     * Menghitung total jumlah baris di tabel film.
     *
     * @return int Jumlah total film.
     */
    public function countAll() {
        $result = $this->conn->query("SELECT COUNT(*) FROM {$this->table}");
        if (!$result) {
            error_log("SQL Error in Film::countAll: " . $this->conn->error);
            return 0;
        }
        $row = $result->fetch_row();
        return $row[0];
    }
}